/*************************************************************************************************************
GPS Clock.c

This is the source code file for the GPS Clock

Geoff Graham
    Version 1.0   Oct 2008
    Version 1.1   Nov 2008 - debugged version
    Version 1.2   Jan 2009 - minor bug fixes
    Version 1.3   Jan 2009 - modified to remove the chance that an interrupt might change variables while the
                             mainline code is using them
    Version 1.4   Mar 2009 - Modified to allow the to be set to -12 or +12 hours
    Version 1.5   Apr 2009 - Fixed major bug affecting startup between 9 and 12 o'clock and other changes
    Version 1.5b  Apr 2009 - Fixed bug in compensating for an accurate crystal.
    Version 2.0   Jul 2009 - Added compatibility with modifications to support sweep hands clocks
                             Note that this version is for stepping clocks only.  There is a separate
                             version for sweep hands clocks.
    Version 2.1   Apr 2012 - Fixed a bug affecting the end of DST when the day is the first day of the month.
    Version 2.2   Mar 2013 - 9600 baud version
    Version 2.3   Jan 2017 - Version for Silicon Chip (RA0, RA1 and RA6 driven in parallel)
    Version 2.5   Mar 2017 - Fix bug with driving RA0, RA1 and RA6 in parallel and bug in GPS parsing
    Version 2.6   Mar 2017 - Fix bug in v2.5, driving wrong output pins

Development Environment
   To compile this you need:
    - CCS C Compiler Version 4.0 or higher (www.ccsinfo.com)
    - Microchip MPLAB IDE Version 8.0 or higher (www.microchip.com)

Program summary:
    The clock's hands are controlled by two variables; ClkTarget (where we want the clock's hands to be pointing)
    and ClkActual (the actual position of the hands).  Both are 32 bit signed integers representing seconds since
    1st Jan 2000.
   
    If ClkActual = ClkTarget -1 then we can step the clock's hands once in a second.  If ClkActual < ClkTarget -1
    then we need to double step to catch up.  And finally, if ClkActual >= ClkTarget then we skip a second.
   
    The skipping and double stepping is performed in TIMER1_isr() which is the timer 1 interrupt.  ClkActual is 
    incremented by this interrupt when a pulse is issued to the clock's stepper motor.  If we want to change the
    clock's hands we change ClkTarget and the comparison in the interrupt routine will work to eventually make 
    the hands point to where ClkTarget is pointing.  For example, at the start of DST we only need to add 
    3600 seconds to ClkTarget and eventually the clock's hands will catch up.
    
    In the timer 1 interrupt routine a number of functions occur:
      - At the start of a new second ClkTarget in incremented by one because a new second has occurred.
      - Also at the start of a new second the code adjusts ClkTarget for a crystal error and daylight saving. 
      - It then figures out if it needs to skip or double step clock pulse.
      - The rest of the interrupt code is controlled by a state machine which is used to work out if the interrupt
        represents the leading or trailing edge of the clock pulse, if we are skipping a pulse or double stepping, etc.
      
    In main() the initialisation code runs, then the CPU enters sleep.  The interrupt will cause the CPU to exit sleep 
    and the following code will check if it needs to do anything (stop on battery low, synchronise with the GPS, etc).  
    Following this the CPU will promptly re enter sleep.
   
    The key function here is SyncTimeWithGPS() which powers up the GPS, checks battery voltage, gets a valid message
    from the GPS, converts the time/date to seconds since 1 Jan 2000, applies the timezone offset, calculates the
    error in the crystal oscillator and sets up the variables to apply compensation for this error, calculates when 
    DST starts/ends, and finally applies compensation if we are in DST.
   
***************************************************************************************************************/

// Defines to control debugging aspects - COMMENT OUT ALL FOR PRODUCTION
//#define Development                   // Use internal clock for timer 1 (no Xtal on pins 13 & 12 so the ICD can be used)
//#define UseDebuggingMarks             // Include diagnostic pulses on unassigned pins for debug purposes
//#define ExcludeUserConfigMenus        // Do NOT include the user configuration menus (more space for debugging code)

#define SiliconChipVersion              // compile for SC's new version of PCB with three pins driving the clock
#opt 9

#define CHECK_CHECKSUM                // Validate the checksum on data from the GPS.  Comment out to save 40 bytes

#define _16F88

#ifdef _16F88
   #include <16F88.h>
   #device ADC=8                       // read just 8 bits from the ADC
   #fuses INTRC_IO,NOIESO,NOPUT,NOFCMEN,NOWDT,NOCPD,NOWRT,NOPROTECT,MCLR,NOBROWNOUT,NOLVP,NODEBUG
   // Clock Related Fuses (select one):
   //        LP  Low-Power Crystal
   //        XT  Crystal/Resonator
   //        HS  High-Speed Crystal/Resonator
   //        RC  External Resistor-Capacitor (RC) with FOSC/4 output on RA6
   //        RC_IO  External Resistor-Capacitor with I/O on RA6
   //        INTRC - Internal Oscillator with FOSC/4 output on RA6 and I/O on RA7
   //        INTRC_IO - Internal Oscillator with I/O on RA6 and RA7
   //        EC_IO  External Clock with I/O on RA6.
   // Int/Ext Clock Switchover:     IESO,NOIESO
   // Fail-Safe Clock Monitor:      FCMEN,NOFCMEN
   // Watch Dog Timer:              WDT, NOWDT
   // Protect Data Memory:          CPD,NOCPD
   // Protect Code:                 PROTECT,NOPROTECT
   // Prog mem write protected      WRT,NOWRT
   // Use MCLR Pin for Reset:       MCLR, NOMCLR
   // Power-up Timer:               PUT,NOPUT
   // Brown-out Detect:             BROWNOUT,NOBROWNOUT
   // Low-Volt Programming Enable:  LVP, NOLVP
   // In-Circuit Debugger Mode:     DEBUG,NODEBUG
   // CCP1 Pin Selection:           CCPB0, CCPB3

   // processor specific hardware registers
   #byte indf = 0               // Indirect register
   #byte tmr0 = 1               // timer0 count register
   #byte t1con = 0x10            // Timer 1 control register
   #byte tmr1H = 0x0f            // timer1 count register high byte
   #byte tmr1L = 0x0e            // timer1 count register low byte
   #byte status = 3            // The status register
   #byte fsr = 4               // Index to RAM
   #byte porta = 5               // Port A I/O register
   #byte portb = 6               // Port B I/O register
   #byte pclath = 0x0a            // Program counter high bits
   #byte intcon = 0x0b            // Basic interrupt control
   #byte cmcon = 0x9c            // Analog comparator control
   #byte rcsta = 0x18            // UART receive status and control
   #byte rcreg = 0x1a            // UART data received register

   #use delay(clock=4000000)
#endif

// define some easy to remember types
#define bit int1
#define word unsigned int16
#define forever  1

// include some standard CCS C Compiler include files
// these should be automatically installed with the C compiler
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#use fast_io(A)   
#use fast_io(B)

#ifdef SiliconChipVersion
  #define TRISA             0b00100111      // this MUST agree with the #defines below
#else
  #define TRISA             0b01100111      // this MUST agree with the #defines below
#endif
#define TRISB               0b11111100      // this MUST agree with the #defines below

#define   CLK_P1              pin_A0          // input - step pulse #2 to the clock movement (set to output later)
#define   CLK_P2              pin_A1          // input - step pulse #1 to the clock movement (set to output later)
#define REF_VOLT            pin_A2          // input  - analog input from the MAX756 reference voltage
#define LED                 pin_A3          // output - simple diagnostic indicator
#define GPS_PWR             pin_A4          // output - low will apply power to the GPS
#define MCLR                pin_A5          // input  - reset circuit used to delay CPU start by 1 second
#ifdef SiliconChipVersion
  #define CLK_P3            pin_A6          // step pulse #3 to the clock movement (set to output later)
#else
  #define GPS_TX            pin_A6          // output - transmit data to the GPS (not used)
#endif
#define PC_TX               pin_A7          // output - RS232 transmit data for setup

#define UNASSIGNED1         pin_B0          // output
#define UNASSIGNED2         pin_B1          // output
#define UART_RX             pin_B2          // input  - mods can link this to pin_B4 (set to output later)
#define PC_RX               pin_B3          // input  - RS232 receive data for setup
#define GPS_RX              pin_B4          // input  - receive data from the GPS
#define BUTTON              pin_B5          // input  - pushbutton input
#define T1OSC1              pin_B6          // input  - reserved for timer 1 oscillator crystal
#define T1OSC2              pin_B7          // input  - reserved for timer 1 oscillator crystal

#define ANALOG_PINS         sAN2            // ADC input ports for reading the MAX756 battery voltage
#define   ANALOG_CHANNEL      2               // the ADC channel for reading the MAX756 battery voltage

#byte      TRISA_REG   = 0x85
#bit      P1_FLOAT   = TRISA_REG.0
#bit      P2_FLOAT   = TRISA_REG.1
#bit      P3_FLOAT   = TRISA_REG.6

// output a pulse on an unassigned pin, used only for debugging
#ifdef UseDebuggingMarks
   #define Mark1            { output_high(UNASSIGNED1); output_low(UNASSIGNED1); }
   #define Mark1double         { output_high(UNASSIGNED1); output_low(UNASSIGNED1); output_high(UNASSIGNED1); output_low(UNASSIGNED1); }
   #define Mark2            { output_high(UNASSIGNED2); output_low(UNASSIGNED2); }
   #define Mark2double         { output_high(UNASSIGNED2); output_low(UNASSIGNED2); output_high(UNASSIGNED2); output_low(UNASSIGNED2); }
   #define Mark3            { output_high(UNASSIGNED3); output_low(UNASSIGNED3); }
   #define Mark3double         { output_high(UNASSIGNED3); output_low(UNASSIGNED3); output_high(UNASSIGNED3); output_low(UNASSIGNED3); }
#else
   #define Mark1            {  }
   #define Mark1double         {  }
   #define Mark2            {  }
   #define Mark2double         {  }
   #define Mark3            {  }
   #define Mark3double         {  }
#endif


// select the baudrate (uncomment the ones that you want)
//#define BAUD4800          // setup console is 4800 baud
//#define GPS_BAUD4800      // GPS is 4800 baud

#define BAUD9600            // setup console is 9600 baud
#define GPS_BAUD9600        // GPS is 9600 baud
#define INVERT_SERIAL

#ifdef BAUD4800
    #ifdef INVERT_SERIAL
        #use rs232(stream=PC, baud=4800, bits=8, parity=N, FORCE_SW, stop=2, xmit=PC_TX, rcv=PC_RX)
    #else
        #use rs232(stream=PC, invert, baud=4800, bits=8, parity=N, FORCE_SW, stop=2, xmit=PC_TX, rcv=PC_RX)
    #endif
#endif

#ifdef BAUD9600
    #ifdef INVERT_SERIAL
        #use rs232(stream=PC, baud=9600, bits=8, parity=N, FORCE_SW, stop=2, xmit=PC_TX, rcv=PC_RX)
    #else
        #use rs232(stream=PC, invert, baud=9600, bits=8, parity=N, FORCE_SW, stop=2, xmit=PC_TX, rcv=PC_RX)
    #endif
#endif


#ifdef SiliconChipVersion
   #ifdef GPS_BAUD4800
       #use rs232(stream=GPS, baud=4800, bits=8, parity=N, FORCE_SW, xmit=UNASSIGNED1, rcv=GPS_RX)
   #endif
   
   #ifdef GPS_BAUD9600
       #use rs232(stream=GPS, baud=9600, bits=8, parity=N, FORCE_SW, xmit=UNASSIGNED1, rcv=GPS_RX)
   #endif
#else
   #ifdef GPS_BAUD4800
       #use rs232(stream=GPS, baud=4800, bits=8, parity=N, FORCE_SW, xmit=GPS_TX, rcv=GPS_RX)
   #endif
   
   #ifdef GPS_BAUD9600
       #use rs232(stream=GPS, baud=9600, bits=8, parity=N, FORCE_SW, xmit=GPS_TX, rcv=GPS_RX)
   #endif
#endif
/************************************************************************************************
Configuration defines
************************************************************************************************/

#define SECONDS_IN_HOUR      3600

#define   GET_GPS_TIMEOUT      240               // number of seconds to wait for a GPS lock
#define GPS_RETRY_TIME      4               // number of hours before retrying a failed GPS lock
#define MAX_GPS_ERRORCOUNT   10               // number of times the GPS can fail to lock before we give up
#define MAX_GPS_DEADCOUNT   10               // number of times the GPS can fail to respond before halting

#define USE_CLK_P2P         128               // threshold to start using the second clock output (128=2.5V, 147=2.25V)
#define LOW_BATT            166               // low battery threshold (166 = 2V) Note: Higher nbr means lower volts
#define MAX_BATT_LOWCOUNT   4               // number of battery low indications before halting

#define DST_OFFSET         SECONDS_IN_HOUR      // the actual DST shift in seconds
#define   HOUR_DST_STARTS      2               // the hour that daylight saving starts
#define   HOUR_DST_ENDS      3               // the hour that daylight saving ends
#define   DST_DAY_OF_WEEK      1               // day of the week that DST starts/ends on
                                                // 0=Sat, 1=Sun, ... Fri=6
                                    

/************************************************************************************************
Global memory locations
************************************************************************************************/

// all variables marked volatile may be changed in a clock interrupt
// and special care must be taken in accessing them as they may change while being used
volatile signed int32 ClkActual;                // the actual physical position of the clock's hands
volatile signed int32 ClkTarget;                // the target position.  We change this variable to change the time
volatile signed int32 XtalSec;                   // the number of seconds by the crystal oscillator
volatile signed int32 utc;                       // the current UTC time
volatile word ClkPos;                          // a 16 bit counter of the clocks hands - used to save program ROM
volatile signed int32 GPScount;                   // countdown to the next GPS run
volatile bit ClkNewSecond;
#ifdef Development
   volatile word GpsTimeout;                    // general purpose timer decremented every second
#else
   volatile byte GpsTimeout;                    // general purpose timer decremented every second
#endif

// these variables are not changed during an interrupt
signed int32 utcLast;                          // the UTC time when last synchronised
bit GotFirstGPSLock = false;                   // true if we got a GPS lock on power up
                                        // happens every second
signed int32 XtalError;                          // >0 if crystal is fast, <0 means the crystal is slow

struct DateTimeStruct {                         // holds the time/date returned by the GPS
   byte hour, min, sec;
   byte day, mth, year;
   word totaldays;
   } GPSdata;   

byte BatteryLowCount;                         // number of times the battery voltage has been low
byte GpsDeadCount;                                  // number of times the GPS did not startup
byte GpsErrorCount;                                 // number of times failed in getting a GPS lock

bit ModifiedPCB;                                    // true if this is running on a board modifies for a sweep hands clock

signed int32 DSTStart, DSTEnd;                      // calculated start/end of DST in seconds

signed int32 TimeZone32;                      // offset from UTC in seconds 

int tab_count;                                      // variable used to count spaces to tab the menu

char MenuIndex;                                     // current menu command char (auto incremented by PrintMenuItem())
char sbuf[20];                                      // buffer for saving strings

// RAM storage for the various configuration values
// these are defined together to make it easy to save them in eeprom
#define CONFIG_DATA_START 0x30
#define CONFIG_DATA_SIZE 8
signed int8 TimeZone;                         // offset from UTC in hours * 10
#locate TimeZone = CONFIG_DATA_START
signed int8 DSTenabled;                         // true if DST is used
#locate DSTenabled = CONFIG_DATA_START + 1
signed int8 DSTmthStart;                      // month that DST starts in
#locate DSTmthStart = CONFIG_DATA_START + 2
signed int8 DSTweekStart;                      // week of the month that DST starts in
#locate DSTweekStart = CONFIG_DATA_START + 3
signed int8 DSTmthEnd;                         // month that DST ends in
#locate DSTmthEnd = CONFIG_DATA_START + 4
signed int8 DSTweekEnd;                         // week of the month that DST ends in
#locate DSTweekEnd = CONFIG_DATA_START + 5
signed int8 ClkPulseWidth;                      // in the range of 2 to 8 (2 = 16mS, 8 = 64mS)
#locate ClkPulseWidth = CONFIG_DATA_START + 6
signed int8 GPScountMax;                      // GPS sync intervals in hours
#locate GPScountMax = CONFIG_DATA_START + 7


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// initialise the eeprom configuration data
// these are the default configuration values set to NSW/Vic/Tas values
#rom  0x2100 = {100,                            // offset from UTC in hours * 10 (e.g.  10 hours = 100) 
            1,                              // true if DST is used
            10,                             // month that DST starts in
            1,                            // week of the month that DST starts in
            4,                            // month that DST ends in
            1,                            // week of the month that DST ends in
            5,                            // clock pulse width in the range of 2 to 8 (2 = 16mS, 8 = 64mS)
            44                           // GPS sync intervals in hours
         }                 

// to save space we also store strings in the eeprom
#define MSG_START 8                                 // offset for the start of messages
#rom  0x2108 =  {"",                                // 0
             " = Set ",                         // 1
             "(now ",                           // 2
             ")\r",                             // 3
             "hours",                           // 4
             "timezone (hrs)",                  // 5
             "daylight saving",                 // 6
             "on",                              // 7
             "mSec",                            // 8
             "start ",                          // 9
             "month",                           // 10
             "Sunday",                          // 11
             "clock pulse (mSec)",              // 12
             "GPS update (hrs)",                // 13
             "end ",                            // 14
             "Enter ",                          // 15
             "\r  Q = Quit\r\rCommand: ",       // 16
             "off",                             // 17
             "-12.0 to +12.0 hrs",              // 18
             "1 to 12",                         // 19
             "1 is 1st, 2 is 2nd, ... 4 is last",   // 20
             " (",                              // 21
             "): ",                             // 22
             "\r",                              // 23
             "\007\r\rInvalid number\r\r"      // 24
            }            
// Hex 2200 is the maximum rom address

#define M_null      0
#define M_set       1
#define M_current   2
#define M_breturn   3
#define M_hour       4
#define M_timezone   5
#define M_dst       6
#define M_on      7
#define M_mSec       8
#define M_start       9
#define M_month       10
#define M_sunday   11
#define M_pulse      12
#define M_update   13
#define M_end      14
#define M_enter      15
#define M_command   16
#define M_off      17
#define M_11to11   18
#define M_1to12      19
#define M_1to4      20
#define M_openb      21
#define M_closeb   22
#define M_return   23
#define M_invalid   24





/************************************************************************************************
Declare functions
************************************************************************************************/
signed int32 GPSDateToSeconds();
void FlashLED(byte nbr);
void halt_program();
void HandlePushbutton();
byte BattVoltage();
bit GetGPSData(bit FullData);
void GpsSkipComma(byte nbr);
byte GetGpsDataByte();
void CheckGpsChar(byte c);
#ifdef CHECK_CHECKSUM
    void CheckGpsHexChar(byte c);
#endif
byte GetGpsChar();
signed int32 CalcDST(byte dsthour, byte dstmonth, byte dstweek);
void AdjustForDST();
void SyncTimeWithGPS();
void DoConfiguration();



/************************************************************************************************
Timer1 interrupt
This does two functions:
 - On each second (NOT each interrupt as we can have multiple interrupts in a second) perform a
   number of tasks related to a new second incl calculate how many clock pulses we want (0,1 or 2)
 - Creates the pulse to step the clock mechanism.  An interrupt occurs on both the leading and
   trailing edges of the pulse (the PIC is in sleep in between) and there might be 2 pulses in a second.
************************************************************************************************/
    static bit ClkRevPulse = false;                 // true if we reverse the clock pulse wire
   
#int_TIMER1
void  TIMER1_isr(void) 
{
    #define sSkip          1
    #define sNewXtalSec    2
    #define sNormEnd       3
    #define sNormStart       4
    #define sDbl2ndEnd       5
    #define sDbl2ndStart    6
    #define sDbl1stEnd       7
    #define sDbl1stStart    8
   
    static byte ClkState = sNewXtalSec;             // variable defining our position in the state machine
    static signed int32 ErrCountdown = 0;           // compensate for the crystal error after this many seconds
    byte Increment;                                 // used to calculate how many clock steps in this second

    // This section handles a new second according to the xtal clock
    // Note that this can differ from the clock stepping code below which might be stepping the clock twice a second
    if(ClkState == sNewXtalSec){
        ClkNewSecond = true;               
        XtalSec++;                                  // a new second according to the xtal clock
        if(GpsTimeout != 0) GpsTimeout--;         // general purpose countdown timer
        if(GPSCount != 0) GPScount--;               // another timer, this is for for the next GPS run
        Increment = 1;                              // how many pulses to the clock mechanism.  may be adjusted below
      
        // This adjusts for an inaccurate crystal frequency.  It adds/subtracts one second every XtalError seconds
        if(ErrCountdown == 0)
            ErrCountdown = XtalError;               // reset our countdown if we reached zero
        if(XtalError > 0) {                         // if the xtal was ahead of the gps time
            if(--ErrCountdown == 0)                 // is it time to make the adjustment?
                Increment--;                        // the result is no increment
        } else if(XtalError < 0){                   // if the xtal was behind the gps time
            if(++ErrCountdown == 0)                 // is it time to make the adjustment?
                Increment++;                        // the result is a double increment
        }         
      
      // apply the increment to the target time plus our record of utc time and check if daylight saving rules apply
      while(Increment--) {
         ClkTarget++; utc++;
         if(DSTenabled) {
            if(utc == DSTStart) ClkTarget += DST_OFFSET;
            if(utc == DSTEnd) ClkTarget -= DST_OFFSET;
         }   
      }
         
      // Now, calculate how to handle the next second.  This can be a normal second which issues one pulse to the
      // clock, or a double pulse if we are catching up, or nothing if we want to loose some time
      if(ClkTarget == ClkActual + 1)
         ClkState = sNormStart;            // we do a normal clock tick
      else
         if(ClkTarget > ClkActual)
            ClkState = sDbl1stStart;      // we must step two clock ticks
         else   // x <= 0
            ClkState = sSkip;            // we must skip a clock tick

   }
   
   
   // This section handles the issuing of stepping pulses to the clock
   // It is mostly independent of the above code and uses a state machine to figure out the pulse timing
   switch(ClkState) {
      case sDbl1stStart:                                 // the start of the first of a double pulse
      case sDbl2ndStart:                                 // the start of the second double pulse
      case sNormStart:                                 // the start of a normal one second pulse
               tmr1H = 256 - ClkPulseWidth;               // set the timer to count the width of the pulse
#ifdef SiliconChipVersion
               if(!ClkRevPulse) {
                 PORTA = PORTA | 0x43;
//                  output_high(CLK_P1);
//                  output_high(CLK_P2);
//                  output_high(CLK_P3);
               } else { 
                 PORTA = PORTA & ~0x43;
//                   output_low(CLK_P1);
//                   output_low(CLK_P2);
//                   output_low(CLK_P3);
               }   
               P1_FLOAT = false;
               P2_FLOAT = false;
               P3_FLOAT = false;
#else
               if(!ClkRevPulse)
                  output_high(CLK_P1);
               else { 
                   output_high(CLK_P2);
                   output_low(CLK_P1);
               }   
               if(ModifiedPCB) P1_FLOAT = false;
#endif
               ClkActual++;                           // update our record of where the clock hands are
               if(++ClkPos == 12 * SECONDS_IN_HOUR) ClkPos = 0;// ditto for a 16 bit int that uses less flash
               ClkState--;
               break;
      case sDbl1stEnd:                                 // end of the first pulse in a double pulse set
      case sDbl2ndEnd:                                 // end of the second pulse in a double pulse set
      case sNormEnd:                                    // end of  of a normal one second pulse
               if(ClkState == sNormEnd)
                  tmr1H = 128 + ClkPulseWidth;            // set up the idle time between normal pulses
               else
                  tmr1H = 192 + ClkPulseWidth;            // and half the time for double pulses
#ifdef SiliconChipVersion
               P1_FLOAT = true;
               P2_FLOAT = true;
               P3_FLOAT = true;
               PORTA = PORTA & ~0x43;
//               output_low(CLK_P1);
//               output_low(CLK_P2);
               output_low(CLK_P3);
#else
               if(ModifiedPCB) P1_FLOAT = true;
               output_low(CLK_P1);
               output_low(CLK_P2);
#endif
               ClkRevPulse = !ClkRevPulse;                  // flip so that the other output does the next pulse
               if(ClkState-- == sDbl2ndEnd) ClkState = sNewXtalSec;
               break;
      case sSkip:                                       // we skip a pulse so the timer is set for a full second
               tmr1H = 128;
               ClkState++;
               break;
   }
}




/**********************************************************************************************
Main program
**********************************************************************************************/

void main() {
   byte i;

    setup_oscillator(OSC_4MHZ);                     // run this chip at 4MHz using the internal clock
   
   output_high(GPS_PWR);                        // set the GPS power control before we enable the outputs
   
   // set pin directions
   set_tris_a(TRISA);
   set_tris_b(TRISB);
   port_b_pullups(0);
   
   
    // set the clock source for Timer 1
#ifdef Development
    setup_timer_1(T1_INTERNAL|T1_DIV_BY_8);
#else
    setup_timer_1(T1_EXTERNAL);
    t1con = 0x0f;
#endif
 
   // initialise output pins
   output_low(CLK_P1);
   output_low(CLK_P2);                           // both clock pins low

   FlashLED(1);                              // Flash the LED once
   
    // check if the PCB has been modified
#ifdef SiliconChipVersion
    ModifiedPCB = true;                                 // assume PCB is modified
   output_low(CLK_P3);                           // both clock pins low
    set_tris_a(TRISA);
#else
    ModifiedPCB = true;                                 // assume PCB is modified
   output_low(CLK_P2);                                 // set A1 low
   set_tris_a(TRISA & 0b11111101);                     // set A1 to an output
   delay_ms(50);
   if(input(CLK_P1) == 0) {                            // did A0 go low?
       output_high(CLK_P2);                            // now try high
       delay_ms(50);
       if(input(CLK_P1) == 1) ModifiedPCB = false;     // if A0 followed it we do NOT have a modified PCB
   }

    // setup according to the state of PCB modification
    if(ModifiedPCB)
       set_tris_a(TRISA);
    else {
       set_tris_a(TRISA & 0b11111100);                 // set A1 and A0 to an output
       output_low(CLK_P1);
       output_low(CLK_P2);                        // both clock pins low
       set_tris_b(TRISB & 0b11111011);                 // set B2 to an output
    }
#endif
        
   // get stored values for the configuration values from EEPROM
   for(i = 0 ; i < CONFIG_DATA_SIZE ; i++) *(CONFIG_DATA_START + i) = read_eeprom(i);

#ifndef ExcludeUserConfigMenus
   if(!input(BUTTON)) DoConfiguration();            // do configuration if the user has held down the button
#endif

   TimeZone32 = (signed int32)TimeZone * 360;         // convert the timezone (in hours times 10) to seconds
   
   output_low(GPS_PWR);                        // power up the GPS
   
   delay_ms(1100);
   if(BattVoltage() < 20)  halt_program();            // check that the MAX756 has powered up
   FlashLED(2);                              // Flash the LED twice to show progress

   delay_ms(1500);
   if(!GetGPSData(false)) halt_program();            // check that the GPS is sending something out
   FlashLED(3);                              // Flash the LED thrice to show that the GPS is alive

   SyncTimeWithGPS();                           // get a valid time from the GPS
   if(!GotFirstGPSLock) halt_program();            // Halt right now if we did not get a GPS lock
   
   GpsDeadCount = 0;
   GpsErrorCount = 0;
   BatteryLowCount = 0;

   ClkTarget++; ClkTarget++;                     // this compensates for the delays in getting started
   ClkActual = (ClkTarget / (12 * SECONDS_IN_HOUR)) * (12 * SECONDS_IN_HOUR); // position of the hands is at 12 o'clock
   ClkPos = 0;                                 // same for our 16 bit counter
   
    // enable interrupts from the crystal based timer
   ClkNewSecond = false;
    tmr1H = 128; tmr1L = 0;                        // set the next timer 1 interrupt in 1 second
    clear_interrupt(INT_TIMER1);
   enable_interrupts(INT_TIMER1);                  // and start processing timer interrupt
    enable_interrupts(GLOBAL);                     // GO !!

   FlashLED(3);                              // Flash the LED four times to show that the GPS got valid data
                                             // Note that the first flash is supplied by SyncTimeWithGPS()
   while(forever) {
      #ifdef Development
         while(!ClkNewSecond) ;
      #else
         while(!ClkNewSecond) sleep();
      #endif
      // we wakeup from sleep and reach here if we have counted a new second
      
      if(GpsDeadCount >= MAX_GPS_DEADCOUNT && ClkPos == 42600) 
         halt_program();                        // GPS is not responding - halt at 11:50
      
      if(GpsErrorCount >= MAX_GPS_ERRORCOUNT && ClkPos == 42900) 
         halt_program();                        // GPS cannot get the time - halt at 11:55
         
      if(BatteryLowCount >= MAX_BATT_LOWCOUNT && ClkPos == 0)
         halt_program();                        // battery low, stop at exactly 12 o'clock
      
      if(GPScount == 0 || !input(BUTTON))            // synch when count is zero or user pressed button
         SyncTimeWithGPS();                     // synchronise

      ClkNewSecond = false;
   }
}



/************************************************************************************************
Get data from the GPS module

Arguments:
FullData   If true the function will return only after receiving a valid $GPRMC string
         If false will return after receiving just "$GP" indicating a working GPS

Returns:   True if data received OK.  In that case struct GPSdata will be filled with data received
         False if timeout
               
Format of the NEMA message we want:
    $GPRMC,043356.000,A,3158.7599,S,11552.8689,E,0.24,54.42,101008,,*20
    ====== ======     =                                     ======   ==
    fixed   time   valid                                    date    checksum
    header hhmmss   data                                    ddmmyy  (ignored)
************************************************************************************************/
bit GpsError;
byte GpsChecksum;

bit GetGPSData(bit FullData) {
    #ifdef CHECK_CHECKSUM
       byte ChecksumSave;
    #endif
   
   #ifdef Development
      GpsTimeout = GET_GPS_TIMEOUT * 16;            // timeout is 16 times higher if we are debugging
   #else                                    // because timer1 is clocked 15 times faster in debugging
      GpsTimeout = GET_GPS_TIMEOUT;               // setup the timeout
   #endif
   
   do {
      GpsError = false;
      while(GetGpsChar() != '$') if(GpsTimeout == 0) return false;
      
      GpsChecksum = 0;
      CheckGpsChar('G');
      CheckGpsChar('P');
      if(!FullData && !GpsError) return true;         // exit if we are just testing if the module is alive
      
      CheckGpsChar('R');
      CheckGpsChar('M');
      CheckGpsChar('C');
      if( GpsError )
        continue; // skip non-RMC messages - important! (NV)
      CheckGpsChar(',');
      GPSdata.hour = GetGpsDataByte();
      GPSdata.min = GetGpsDataByte();
      GPSdata.sec = GetGpsDataByte();
      GpsSkipComma(1);
      CheckGpsChar('A');                        // indicates that the GPS has a lock on enough satellites
      GpsSkipComma(7);
      GPSdata.day = GetGpsDataByte();
      GPSdata.mth = GetGpsDataByte();
      GPSdata.year = GetGpsDataByte();

        #ifdef CHECK_CHECKSUM
          do 
              ChecksumSave = GpsChecksum;
          while(GetGpsChar() != '*');                 // skip to the checksum
          CheckGpsHexChar(ChecksumSave >> 4);         // first byte of the checksum
          CheckGpsHexChar(ChecksumSave & 0x0f);       // and the second
        #else
            CheckGpsChar(',');                          // just test for a comma - very unlikely that it would fail
        #endif
   } while(GpsError);                           // only exit if everything OK
   
   return true;
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a number of commas.  Used for skipping over unwanted data
//
void GpsSkipComma(byte nbr) {
   while(nbr)
      if(GetGpsChar() == ',') nbr--;
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a data byte consisting of two ascii chars
//
byte GetGpsDataByte() {
   byte b;
   
   b = (GetGpsChar() - '0') * 10;
   if(GpsError) return 0;
   b += (GetGpsChar() - '0');
   return b;   
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a character and test it against the argument.  Set GpsError if mismatch.
//
void CheckGpsChar(byte c) {
   if(GetGpsChar() != c) GpsError = true;
}
   
   

#ifdef CHECK_CHECKSUM
///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a hex character and test it against the argument.  Set GpsError if mismatch.
//
void CheckGpsHexChar(byte c) {
    byte ch;
   ch = GetGpsChar() - '0';                  // get the first byte of the checksum
   if(ch > 9) ch -= 'A' - 58;                  // convert from hex
   if(ch != c) GpsError = true;                   // and test
}
#endif   


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get a char from the GPS input stream.  
//   - Checks for timeout and returns zero if exceeded.
//   - Skips the read and returns zero if GpsError already set
//   - Converts the char to uppercase and adds it to the checksum
//
byte GetGpsChar() {
   byte ch;
   
   while(!kbhit(GPS))
      if(GpsError || GpsTimeout == 0) {
         GpsError = true;
         return 0;
      }   
         
   ch = fgetc(GPS);
   GpsChecksum = GpsChecksum ^ ch;
   if(ch >= 'a') ch -= 'a' - 'A';                  // convert to uppercase
   return ch;
}   
   


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Perform all the work involved in getting the time from the GPS and correcting for any errors 
// This includes setting ClkTarget, XtalError and daylight saving comparison variables
//
void SyncTimeWithGPS() {         
   signed int32 t;
   bit GotGPSData;
   
   output_low(GPS_PWR);                        // turn on the GPS
   
   delay_ms(550);                              // wait for the EM-408 module to draw full current
    if(BattVoltage() > LOW_BATT)                        // check battery. Note: Higher number means lower voltage
      BatteryLowCount++;
   else
      BatteryLowCount = 0;

   GPScount = GPS_RETRY_TIME * SECONDS_IN_HOUR;      // this is the default timeout if we have an error
      
   if(!GetGPSData(false)) {                     // check that the GPS is alive
      GpsDeadCount++;
      output_high(GPS_PWR);                     // turn off the GPS
      return;                                 // and skip the sync
   }
   
   GpsDeadCount = 0;                           // GPS is working
   
   GotGPSData = GetGPSData(true);                  // get the date/time from the GPS
   output_high(GPS_PWR);                        // turn off the GPS
   if(!GotGPSData) {
      GpsErrorCount++;
      return;                                 // and abort
   }
   
   GpsErrorCount = 0;
   flashLED(1);                              // show that we got good data
   
   // Wait for a new second.  This sychronisation is necessary because the interrupt may change variables while 
   // the remainder of this function is also changing them.  Following this wait we will have at least 16mS before 
   // the next interrupt occurs and that is plenty of time to complete all the processing in this routine.
   ClkNewSecond = false;
   while(!ClkNewSecond && (intcon & 0b0010000));      // if timer 0 is running, wait for a new second                  
   
   utc = GPSDateToSeconds();                     // the GPS time in seconds since 1 Jan 2000
   if(GotFirstGPSLock) {                        // don't bother if this is the first time sync
      utc++;                                 // compensate for the delay while waiting for the new second
      t = (XtalSec -(utc - utcLast));               // calculate the error factor between the xtal and the gps
      if(t == 0)                              // prevent divide by zero
         XtalError = 0;
      else   
         XtalError = XtalSec / t;
   } else
      XtalError = 0;
         
   utcLast = utc;                              // save the current time for the next gps reading
   GotFirstGPSLock = true;
   
   // reset everything ready for another run
   GPScount = (signed int32)GPSCountMax * SECONDS_IN_HOUR;   // setup the counter for the next sync
   ClkTarget = utc + TimeZone32;                  // Set where the hands of the clock should be pointing
   XtalSec = 0;
   
   // Adjust for daylight saving.  If we are in DST this adds one hour to the clock target.
   // Also, it sets two variables (DSTStart and DSTEnd) that are compared to our count of utc time in the 
   // timer 1 interrupt, if, while incrementing a second, one of these matches we add/subtract one hour for DST.
   if(DSTenabled) {                           // Don't bother if no daylight saving
      
      // first get the start/end times for DST in the current year (held in GPSdata.year)
      // note that DSTEnd will occur during DST therefor, to get a correct UTC, we must subtract the DST offset
      DSTStart = CalcDST(HOUR_DST_STARTS, DSTmthStart, DSTweekStart);
      DSTEnd = CalcDST(HOUR_DST_ENDS, DSTmthEnd, DSTweekEnd) - DST_OFFSET;
      
      // if both start and end are less than the current time, recalculate the earliest one for next year
      // this will convert:
      //          Start < End <  Now      -to-        End  <  Now  < Start (next year)
      //          End  < Start < Now      -to-        Start < Now  < End (next year)
      GPSdata.year++;
      if(DSTStart < utc && DSTEnd < ClkTarget) {
         if(DSTStart < DSTEnd)
            DSTStart = CalcDST(HOUR_DST_STARTS, DSTmthStart, DSTweekStart);
         else
            DSTEnd = CalcDST(HOUR_DST_ENDS, DSTmthEnd, DSTweekEnd) - DST_OFFSET;
      }      
      
      // figure out if we are currently in the daylight saving period and apply the offset
      // we have four possibilities:
      //         Start <= Now   <  End         We are in DST
      //         End   <  Now   <  Start         We are not in DST
      //          Now   <  Start <  End         We are not in DST
      //         Now   <  End   <  Start         We are in DST
      if((DSTStart <= utc && utc < DSTEnd) || (utc < DSTEnd && DSTEnd < DSTStart))
         ClkTarget += DST_OFFSET;
   }
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Convert the GPS date/time (stored in GPSdata) to number of seconds since 1/1/00
// Also fills GPSdata.totaldays with the nbr of days since 1/1/00
//
signed int32 GPSDateToSeconds() {
   word   tt;
   
   switch(GPSdata.mth) {
      case 1:   GPSdata.totaldays = 0; break;
      case 2:   GPSdata.totaldays = 31; break;
      case 3:   GPSdata.totaldays = 31 + 28; break;
      case 4:   GPSdata.totaldays = 31 + 28 + 31; break;
      case 5:   GPSdata.totaldays = 31 + 28 + 31 + 30; break;
      case 6:   GPSdata.totaldays = 31 + 28 + 31 + 30 + 31; break;
      case 7:   GPSdata.totaldays = 31 + 28 + 31 + 30 + 31 + 30; break;
      case 8:   GPSdata.totaldays = 31 + 28 + 31 + 30 + 31 + 30 + 31; break;
      case 9:   GPSdata.totaldays = 31 + 28 + 31 + 30 + 31 + 30 + 31 + 31; break;
      case 10:GPSdata.totaldays = 31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30; break;
      case 11:GPSdata.totaldays = 31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31; break;
      case 12:GPSdata.totaldays = 31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31 + 30; break;
   }
   
   GPSdata.totaldays += (word)GPSdata.year * 365 + ((GPSdata.year - 1)/4) + (word)GPSdata.day;
   if(GPSdata.mth > 2 && (GPSdata.year % 4) == 0) GPSdata.totaldays++;      // leap year. Ignore centuries as the next is in 90+ years
   
   tt = (word)GPSdata.sec + ((word)GPSdata.min * 60);
   return ((signed int32)GPSdata.hour * SECONDS_IN_HOUR) + ((signed int32)GPSdata.totaldays * (24 * SECONDS_IN_HOUR)) + tt;
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Calculate the seconds since 1/1/00 when daylight saving starts or ends in the current year
// Note:  This relies on GPSdata.year being pre filled with the year (getting data from the GPS does that)
//
signed int32 CalcDST(byte dsthour, byte dstmonth, byte dstweek) {
   signed int32 t, tt;
   byte j, ty;
   
   GPSdata.sec = GPSdata.min = 0; GPSdata.hour = dsthour;   // set the date/time for the DST event
   GPSdata.day = 1; GPSdata.mth = dstmonth;            // the year was filled in by the GPS
   t = GPSDateToSeconds();                           // convert to seconds since 1/1/2000
   j = GPSdata.totaldays % 7;                        // get the day of the week with sat = 0
   j = 7 - j + DST_DAY_OF_WEEK; if(j >= 7) j -= 7;         // number of days to the first DST_DAY_OF_WEEK (Sunday = 1)
   j += (dstweek - 1) * 7;                           // number of days to when DST starts
   t += (signed int32)j * 24 * SECONDS_IN_HOUR;         // this is the start time
   if(dstweek == 4) {                              // this is a special case.  4 can also mean the last in the month
      tt = t + 7 * 24 * SECONDS_IN_HOUR;               // add another week - will this still be in the month?
      ty = GPSdata.year;                           // save the year in case we change it below
      if(++GPSdata.mth > 12) {                     // move to the next month
         GPSdata.mth = 1; GPSdata.year++;            // adjust if it pushed us into a new year
      }
      if(tt < GPSDateToSeconds()) t = tt;               // retain the new value if it is still in the month
      GPSdata.year = ty;                           // replace the year in case we changed it
   }
   return t - TimeZone32;                           // return the time in UTC (or GPS time)
}      
      
               


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Flash the LED a number of times
// used for simple diagnostics
//
void FlashLED(byte nbr) {
   byte i;
   
   for(i = 0 ; i < nbr ; i++) {
      delay_ms(200);
      output_high(LED);
      delay_ms(100);
      output_low(LED);
   }
}      


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// This will disable all sections of the MCU and put it to sleep
// It is a low power shutdown (less than 1 microamp), there is no return
//
void halt_program() {
   delay_ms(150);                        // wait for the clock to step the last tick
   disable_interrupts(INT_TIMER1);
   disable_interrupts(INT_RDA);
   setup_timer_1(T1_DISABLED);
   P1_FLOAT = true;
   P2_FLOAT = true;
#ifdef SiliconChipVersion
   P3_FLOAT = true;
#endif
   output_high(GPS_PWR);                  // turn off the GPS
   while(forever) sleep();                  // and go to sleep forever (until the battery is disconnected)
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Read the reference voltage from the MAX756
// Because the ADC reference is the battery voltage we can use this reading to infer the battery voltage.
// This means that decreasing battery voltage will give increasing readings.  With MAX756 Vref = 1.27V a reading
// of 166 gives a battery is 2.0V  A reading of zero means that the MAX756 is not running.
//
byte BattVoltage() {
   byte t;

   setup_adc(ADC_CLOCK_INTERNAL );               // Sets up the a/d mode
   setup_adc_ports(ANALOG_PINS);               // Sets the available adc pins to be analog
   
   set_adc_channel(ANALOG_CHANNEL);             // Select the channel (we only have one anyway)
   delay_us(10);                           // wait for the input to settle
   t = read_adc();                           // read the voltage
   
   #ifdef 16F688
   set_tris_C(TRIS_C);                        // the ADC seems to interfere with TRIS
   #endif
   
   setup_adc(ADC_OFF );
   return t;
}   



/************************************************************************************************
Functions associated with changing the configuration data in EEPROM

These can be removed for testing - this leaves more memory for printf's, etc.
To do so uncomment #define ExcludeUserConfigMenus from the start of the code.

Because we are so short of ROM, many strings are stored in EEPROM and functions like
FindString(int msg_nbr) are used to retrieve them.
************************************************************************************************/

#ifndef ExcludeUserConfigMenus                                       // more space for debugging

///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Find a string in the EEPROM
//
int FindString(int msg_nbr) {
   int i;
   for(i = MSG_START ; msg_nbr != 0 ; i++) if(read_eeprom(i) == 0)
      msg_nbr--;                                                // get the message
   return i;
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Copy a string into the string buffer
//
void CopyString(int msg_nbr) {
   int i, j;
   j = 0;
   for(i = FindString(msg_nbr); (sbuf[j] = read_eeprom(i)) != 0; i++) j++;      // copy the string
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Output a character to the user and flash the LED
//
void SendCharWithFlash(int c) {
   output_high(LED);
   fputc(c, PC);
   output_low(LED);
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Output a string to the user
//
void OutputString(int msg_nbr) {
   int i, c;
   for(i = FindString(msg_nbr); read_eeprom(i) != 0; i++) {               // output the string
      c = read_eeprom(i);
      SendCharWithFlash(c); tab_count++;
      if(c == '\r') fputc('\n', PC);
   }
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Print a menu item
// It outputs in sequence:
//   - a couple of spaces
//   - The menu number contained in the global MenuIndex
//   - The string " = Set "
//   - The FIRST argument
//   - A space
//   - The SECOND argument
//  - Tab to the 33rd character position
//   - An opening bracket
//   - The contents of sbuf[] which is normally the current value
//   - A closing bracket
void PrintMenuItem(txt1, txt2, txt3) {
   int i;
    
    tab_count = 4;
   SendCharWithFlash(' ');
   SendCharWithFlash(' ');
   fputc(MenuIndex, PC);
   OutputString(M_set);      // = Set 
   OutputString(txt1);
   OutputString(txt2);
   fputc(' ', PC);
   OutputString(txt3);
   while(tab_count < 45) { SendCharWithFlash(' ');  tab_count++;}
   OutputString(M_current);         // (
   for(i = 0; sbuf[i] != 0; i++) SendCharWithFlash(sbuf[i]);
   OutputString(M_breturn);         // )/r
   MenuIndex++;
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get a number from the user
// value = pointer to a signed int8.  If the number is valid it will be copied into this location
// lo    = lowest acceptable number
// lo    = highest acceptable number
// point = true if nn.n is acceptable (only used for the timezone).  For example 8 is returned as 80, 8.5 as 85
// txt1, txt2 and txt3 = the numbers of strings to output as a prompt
// txt4    = further informative information for the user, will be outputted between brackets
void GetNumber(signed int8 *value, signed int8 lo, signed int8 hi, bit point, byte txt1, byte txt2, byte txt3, byte txt4) {
   byte ch;
   signed int16 nbr;
   bit neg;
   bit gotpoint;
   bit gotnbr;
   
   OutputString(M_enter);      // = Enter 
   OutputString(txt1);
   OutputString(txt2);
   SendCharWithFlash(' ');
   OutputString(txt3);
   if(txt4 != 0) {
      OutputString(M_openb);
      OutputString(txt4);
      OutputString(M_closeb);
   }
   nbr = 0; gotnbr = gotpoint = neg = false;
   while(forever) {
      SendCharWithFlash(ch = fgetc(PC));
      if(ch == '-')
         neg = true;
      else if(ch == '+')
         ;
      else if(ch == '.' && gotpoint == false && point)
         gotpoint = true;
      else if(ch >= '0' && ch <= '9')
         {nbr = nbr * 10 + (ch - '0'); gotnbr = true; }
      else if(ch == '\r') {
         if(neg) nbr = -nbr;
         if(point && !gotpoint) nbr *= 10;
         if(nbr >= lo && nbr <= hi && gotnbr) {
            *value = nbr;
            OutputString(M_return);
            return;
         } else
            goto InpErr;
      } else
         goto InpErr;
   }
   InpErr:
   OutputString(M_invalid);      
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Print the menu and get input from the user.  Updates the EEPROM accordingly
// Can return to the caller if the user enters Q
//
void DoConfiguration() {
   char ch;
   byte i;
   word j;
   
   fprintf(PC, "\r\n\nGPS Clock firmware 2.6\r\n");
   while(forever) {
      //OutputString(M_restartmsg);
       MenuIndex = '1';
      OutputString(M_return); 
      OutputString(M_return); 
      
       if(TimeZone >= 0) {
          sbuf[0] = '+'; i = TimeZone;
       } else {
          sbuf[0] = '-'; i = -TimeZone;
       }
       sprintf(sbuf+1, "%u", i);
       j = strlen(sbuf);
       if( j == 2 ) {
         sbuf[4] = '\0';
         sbuf[3] = sbuf[1];
         sbuf[2] = '.';
         sbuf[1] = '0';
       } else {
         sbuf[j+1] = '\0';
         sbuf[j] = sbuf[j-1];
         sbuf[j-1] = '.';
       }                                              PrintMenuItem(M_null, M_timezone, M_null);
//       sprintf(sbuf+1, "%3.1w", i);                 PrintMenuItem(M_null, M_timezone, M_null);

       sbuf[0] = 'o'; 
       if(DSTenabled) {
          sbuf[1] = 'n'; sbuf[2] = 0; 
          PrintMenuItem(M_null, M_dst, M_off);
          fputc('\n', PC);
          sprintf(sbuf, "%u", DSTmthStart);              PrintMenuItem(M_start, M_dst, M_month);
          sprintf(sbuf, "%u", DSTweekStart);              PrintMenuItem(M_start, M_dst, M_sunday);
          sprintf(sbuf, "%u", DSTmthEnd);              PrintMenuItem(M_end, M_dst, M_month);
          sprintf(sbuf, "%u", DSTweekEnd);              PrintMenuItem(M_end, M_dst, M_sunday);
        } else {
          sbuf[1] = 'f'; sbuf[2] = 'f'; sbuf[3] = 0; 
          PrintMenuItem(M_null, M_dst, M_on);
       }
       fputc('\n', PC);
       sprintf(sbuf, "%u", ClkPulseWidth<<3);              PrintMenuItem(M_null, M_pulse, M_null);
       sprintf(sbuf, "%u", GPScountMax);                PrintMenuItem(M_null, M_update, M_null);
       
      fprintf(PC, "\r\n  R = Run for one hour");
       OutputString(M_command);
       ch = fgetc(PC); SendCharWithFlash(ch); OutputString(M_return);
       if(!DSTenabled && ch > '2') ch += 4;
       switch(ch) {
          case '1':   GetNumber(&TimeZone, -120, 120, true, M_timezone, M_null, M_null, M_11to11);
                   break;
          case '2':   DSTenabled = !DSTenabled;
                   break;
          case '3':   GetNumber(&DSTmthStart, 1, 12, false, M_start, M_dst, M_month, M_1to12);
                   break;
          case '4':   GetNumber(&DSTweekStart, 1, 4, false, M_start, M_dst, M_sunday, M_1to4);
                   break;
          case '5':   GetNumber(&DSTmthEnd, 1, 12, false, M_end, M_dst, M_month, M_1to12);
                   break;
          case '6':   GetNumber(&DSTweekEnd, 1, 4, false, M_end, M_dst, M_sunday, M_1to4);
                   break;
          case '7':   ClkPulseWidth = ClkPulseWidth<<3;
                   GetNumber(&ClkPulseWidth, 16, 96, false, M_null, M_pulse, M_null, M_null);
                   ClkPulseWidth = ClkPulseWidth>>3;
                   break;
          case '8':   GetNumber(&GPScountMax, 1, 125, false, M_null, M_update, M_null, M_null);
                   break;
          case 'r'+4:
          case 'R'+4:
          case 'r':
          case 'R':   fprintf(PC, "\r\nPress Setup button to abort");
                        for(j = SECONDS_IN_HOUR; j != 0; j--) {
#ifdef SiliconChipVersion
                       if(!ClkRevPulse) {
                        PORTA = PORTA | 0x43;
//                        output_high(CLK_P1);                  // do the leading edge of the pulse
//                        output_high(CLK_P2);
//                        output_high(CLK_P3);
                     } else {
                        PORTA = PORTA & ~0x43;
//                        output_low(CLK_P1);
//                        output_low(CLK_P2);
//                        output_low(CLK_P3);
                     }   
                       output_drive(CLK_P1);
                       output_drive(CLK_P2);
                       output_drive(CLK_P3);
                     delay_ms(ClkPulseWidth<<3);
                       output_float(CLK_P1);
                       output_float(CLK_P2);
                       output_float(CLK_P3);
                     PORTA = PORTA & ~0x43;
//                     output_low(CLK_P1);                             // then the trailing edge 
//                     output_low(CLK_P2);
//                     output_low(CLK_P3);
#else
                       if(!ClkRevPulse) 
                        output_high(CLK_P1);                  // do the leading edge of the pulse
                     else {
                        output_high(CLK_P2);
                        output_low(CLK_P1);
                     }   
                       if(ModifiedPCB) output_drive(CLK_P1);
                     delay_ms(ClkPulseWidth<<3);
                       if(ModifiedPCB) output_float(CLK_P1);
                     output_low(CLK_P1); output_low(CLK_P2);         // then the trailing edge
#endif
                     ClkRevPulse = !ClkRevPulse;                  // flip so that the other output does the next pulse
                     for(i = 1000/8 - ClkPulseWidth; i != 0; i--) {
                        delay_ms(8);
                        if(!input(BUTTON)) goto stop_stepping;
                     }   
                  }
                  stop_stepping:
                  break;
          case 'q'+4:
          case 'Q'+4:
          case 'q':
          case 'Q':   return;
   
       }
      // write the configuration variables (stored in contiguous RAM) to the eeprom
      for(i = 0 ; i < CONFIG_DATA_SIZE ; i++) write_eeprom (i, *(CONFIG_DATA_START + i));
   }      
}

#endif
